; Apple ][ Disk Dumper
; Copyright (c) Sebastian Kienzl <seb at riot dot org>
; $Id: disk2cassette.asm 29 2008-09-15 20:26:26Z seb $
; This thing dumps a whole DOS 3.3 disk and sends it out the
; CASSETTE OUT-port of the Apple. The output can be decoded
; with my a2cassdecode-tool. If the output-file is given the DSK-extension
; it can be directly used with emulators.
;
; assemble with DASM (http://dasm-dillon.sourceforge.net/)
; load to $c00
; run from $c00
;
; this is how it works:
;  * load sectorsPerFlush sectors to the memory starting at $2000
;  * send that data via CASSETTE OUT
;  * loop until all sectors have been loaded
;
; it will display the current read track and sector and the return code,
; 00 is good.
; if it prints a "-" it's just sending via CASSETTE OUT.

	processor   6502

	include "macros.h"

; DOS < 3.3: 13 sektoren / 3.3: 16 sektoren

; ROM Monitor entry points
funcPrintHex		equ		$fdda
funcPrintSpaces		equ		$f948
funcPrintReturn		equ		$fd8e
funcPrintChar		equ		$fdf0
funcBeep			equ		$fbdd
funcTapeWrite		equ		$fecd

; DOS entry points
funcDOSGetIOB		equ		$3e3
funcDOSOperation	equ		$3d9

; this is where the data is placed
bufferStart			equ		$2000
sectorsPerFlush		equ		117

	seg.u	zeropage
	org $e0
IOB			ds	1
IOBH		ds	1

SECTOR		ds	1
TRACK		ds	1

MAXSECTOR	ds	1
MAXTRACK	ds	1

SECTORCOUNT	ds	1

BUFFERH		ds	1		; current buffer-page

pusha_save	ds	1
temp1		ds	1


; MONITOR-Variables
	org		$3c
MON_A1L		ds	1
MON_A1H		ds	1
MON_A2L		ds	1
MON_A2H		ds	1


; ---------------------------------------------
	seg		code
	;org		$2000
	org		$c00
	
	; get IOB-pointer	
	jsr		funcDOSGetIOB
	sty		IOB
	sta		IOBH
	
	jsr		resetBuffer

	; set maximums
	;lda		#13
	lda		#16
	sta		MAXSECTOR
	lda		#35
	sta		MAXTRACK

	; reset
	lda		#0
	sta		TRACK
	lda		#0
	sta		SECTOR

readLoop:
	; output
	lda		TRACK
	jsr		funcPrintHex
	jsr		funcPrintSpaces
	lda		SECTOR
	jsr		funcPrintHex
	lda		#':+$80
	jsr		funcPrintChar
	jsr		funcPrintSpaces

	jsr		readSector
	jsr		funcPrintHex
	jsr		funcPrintReturn	

	; next page
	ldx		BUFFERH
	inx	
	stx		BUFFERH
	
	ldx		SECTOR
	inx
	; >= MAXSECTOR?
	cpx		MAXSECTOR
	bcc		sectorOk
	
	; increment track
	ldx		TRACK
	inx
	; >= MAXTRACK
	cpx		MAXTRACK
	bcs		done
	stx		TRACK

	ldx		#0
sectorOk:
	stx		SECTOR

	ldx		SECTORCOUNT
	dex
	stx		SECTORCOUNT
	bne		noflush
	; we've read maximum sectors that fit, send them
	jsr		flushBuffer
noflush:
	
	jmp		readLoop
	
done:
	jsr		flushBuffer
	
	jsr		funcBeep
	jsr		funcBeep
	jsr		funcBeep

	rts

resetBuffer	subroutine
	lda		#sectorsPerFlush
	sta		SECTORCOUNT
	lda		#>bufferStart
	sta		BUFFERH
	rts

sendTape subroutine
	lda		#<bufferStart
	sta		MON_A1L
	lda		#>bufferStart
	sta		MON_A1H

	; end of buffer -> monitor functions want address-1
	lda		#$ff
	sta		MON_A2L
	ldx		BUFFERH
	dex
	stx		MON_A2H
	
	jsr		funcTapeWrite
	rts

flushBuffer subroutine
	lda		#'-+$80
	jsr		funcPrintChar
	jsr		funcPrintReturn	

	lda		#<bufferStart
	cmp		BUFFERH
	beq		.1
	
	jsr		sendTape
	jsr		resetBuffer

.1
	rts

readSector	subroutine
	pushxy
	
	; save IOB-buffer-destination on stack
	ldy		#8
	lda		(IOB),Y
	pha
	iny
	lda		(IOB),Y
	pha
	
	; set to our buffer
	ldy		#8
	lda		#0
	sta		(IOB),Y
	iny
	lda		BUFFERH
	sta		(IOB),Y
	
	; match any volume
	ldy		#3
	lda		#0
	sta		(IOB),Y
	
	; track number
	ldy		#4
	lda		TRACK
	sta		(IOB),Y
	
	; sector number
	iny
	lda		SECTOR
	sta		(IOB),Y
	
	; read 256 bytes
	ldy		#$b
	lda		#0
	sta		(IOB),Y
	
	; command = read
	iny
	lda		#1
	sta		(IOB),Y

	; load IOB
	ldy		IOB
	lda		IOBH
	jsr		funcDOSOperation
	
	; if carry set, error...
	; save in temp1 until later
	lda		#0
	bcc		no_error
	; return code
	ldy		#$d
	lda		(IOB),Y
no_error:
	sta		temp1

	; restore IOB-buffer-destination
	pla
	ldy		#9
	sta		(IOB),Y
	pla
	dey
	sta		(IOB),Y
	
	popxy

	; "Location $48 should be set to zero after each call to RWTS"
	lda		#0
	sta		$48
	
	; restore return code
	lda		temp1
	
	rts

	nop
